package gov.va.med.mhv.sm.admin.service.impl;

import java.util.List;

import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import gov.va.med.mhv.sm.admin.jpa.model.Clinician;
import gov.va.med.mhv.sm.admin.jpa.repository.ClinicianTriageMapRepository;
import gov.va.med.mhv.sm.admin.jpa.repository.SMUserRepository;
import gov.va.med.mhv.sm.admin.service.util.ResponseCodeUtil;
import gov.va.med.mhv.sm.admin.converter.SMUserConverter;
import gov.va.med.mhv.sm.admin.data.enums.SMErrorEnum;
import gov.va.med.mhv.sm.admin.dto.ClinicianDTO;
import gov.va.med.mhv.sm.admin.exception.SMApiException;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.ArraySchema;

@Path("/clinicians")
@Service("Clinician Service")
@Tag(name = "Clinician Service")
@Consumes(MediaType.APPLICATION_JSON)
@Produces(MediaType.APPLICATION_JSON)
@Component
public class ClinicianService { 

	private static final Log log = LogFactory.getLog(ClinicianService.class);
	
	@Autowired
    private ClinicianTriageMapRepository clinicianTriageMapRepository;
	
	@Autowired
    private SMUserRepository smUserRepository;
	
	@GET
	@Operation(summary = "Get Clinicians based on Query Parameters provided.")
	@Transactional
	@ApiResponses(value = {
	        @ApiResponse(responseCode = "200", 
	        			content = @Content(mediaType = "application/json", 
	        			array = @ArraySchema(schema = @Schema(implementation = ClinicianDTO.class)))),
			@ApiResponse(responseCode = "404", description = "Triagegroup Not found",content = @Content(mediaType = MediaType.APPLICATION_JSON, schema = @Schema(implementation = SMApiException.class))),
			@ApiResponse(responseCode = "500", description = "Unexpected Error Occurred",content = @Content(mediaType = MediaType.APPLICATION_JSON, schema = @Schema(implementation = SMApiException.class)))
	})
	public List<ClinicianDTO> getClinicians(
											@QueryParam("triageGroupId") Long triageGroupId,
											@QueryParam("station") String station) throws SMApiException {
		try{
			ClinicianEnum.SearchEnum clinicianEnum = validateQueryParams(triageGroupId,station);
			switch (clinicianEnum) {
					case GET_CLINICIANS_BY_TRIAGE_GROUP_ID:
						return getCliniciansByTriageGroup(triageGroupId);
					case GET_CLINICIANS_BY_STATION:
						return getCliniciansByStation(station);
					default:
						throw new SMApiException(SMErrorEnum.INVALID_HTTP_REQUEST.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);		
			}
		}catch(Exception clinicianExp){
			log.error(clinicianExp);
			throw new SMApiException(SMErrorEnum.UNKNOWN_EXCEPTION.getErrorMessage(), ResponseCodeUtil.STATUS_INTERNAL_SERVER_ERROR,clinicianExp);
		}
	}
	
	private ClinicianEnum.SearchEnum validateQueryParams(Long triageGroupId, String station) throws SMApiException{
		if(triageGroupId !=null){
			if(station !=null){
				throw new SMApiException(SMErrorEnum.INVALID_HTTP_REQUEST.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);
			}else{
				return ClinicianEnum.SearchEnum.GET_CLINICIANS_BY_TRIAGE_GROUP_ID;
			}
		}else if(station !=null){
			if(triageGroupId !=null){
				throw new SMApiException(SMErrorEnum.INVALID_HTTP_REQUEST.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);
			}else{
				return ClinicianEnum.SearchEnum.GET_CLINICIANS_BY_STATION;
			}
		}
		else{
			throw new SMApiException(SMErrorEnum.INVALID_HTTP_REQUEST.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);
		}
	}
	
	
	private List<ClinicianDTO> getCliniciansByTriageGroup(Long triageGroupId) throws SMApiException {
		List<ClinicianDTO> clinicianDtoList = null;
		List<Object[]> clinicianList = null;
		try 
		{
			if(triageGroupId !=null){
				clinicianList = clinicianTriageMapRepository.getCliniciansByTriageGroup(triageGroupId);
				if(clinicianList !=null){
					clinicianDtoList = SMUserConverter.convertClinicianObjectList(clinicianList);
				}else{
					throw new SMApiException(SMErrorEnum.CLINICIANS_NOT_FOUND.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);
				}
			}
		} catch (Exception clinicianExp) {
			log.error(clinicianExp);
			throw new SMApiException(SMErrorEnum.UNKNOWN_EXCEPTION.getErrorMessage(), ResponseCodeUtil.STATUS_INTERNAL_SERVER_ERROR,clinicianExp);
		}
		return clinicianDtoList;
	}
	
	private List<ClinicianDTO> getCliniciansByStation(String station) throws SMApiException {
		List<ClinicianDTO> clinicianDtoList = null;
		List<Clinician> cliniciansList = null;
		try 
		{
			if(station !=null){
				cliniciansList = smUserRepository.getCliniciansByStation(station);
			}
			if(cliniciansList !=null){
				clinicianDtoList = SMUserConverter.convertClinicianList(cliniciansList);
			}else{
				throw new SMApiException(SMErrorEnum.CLINICIANS_NOT_FOUND.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);
			}
		} catch (Exception clinicianExp) {
			log.error(clinicianExp);
			throw new SMApiException("Unexpected Error Occurred", ResponseCodeUtil.STATUS_INTERNAL_SERVER_ERROR,clinicianExp);
		}
		return clinicianDtoList;
	}
	
	private static class ClinicianEnum{
		public enum SearchEnum{
			GET_CLINICIANS_BY_TRIAGE_GROUP_ID,
			GET_CLINICIANS_BY_STATION;
		}
	}
}
